from data_holders import GatheredOverallData, Settings, AlarmHandler
from display import Display
from picographics import DISPLAY_INKY_PACK
import time
import machine
from mlx90641 import MLX90641
from machine import Pin, PWM
from ina219 import INA219
from data_processing import catch_fire, convert_voltage_to_percent, print_thermal_frame, print_frame_grid
from breakout_bme68x import BreakoutBME68X, STATUS_HEATER_STABLE
from pimoroni_i2c import PimoroniI2C


INA219_COUNTER_VAL          = const(0)
SENSOR_READ_PERIOD          = const(12_000)
CAMERA_DELAY_AFTER_POWER_ON = const(1100)
CPU_FREQ_NORMAL             = const(128_000_000) 
CPU_FREQ_SLEEP              = const(18_000_000) 
BME688_FIRST_WAKEUP         = const(0)
BME688_MID_QUALITY_LEVEL    = const(18_000)

machine.freq(CPU_FREQ_NORMAL)
ina219_counter    = 0
bme_start_counter = 0

buzzer = Pin(2, Pin.OUT, Pin.PULL_DOWN)
buzzer.value(1)
camera_trigger_pin = Pin(11, Pin.OUT, Pin.PULL_DOWN)
camera_trigger_pin.value(1)
time.sleep(2)  # Let camera warm up

# Objects creation
alarm_handler_inst = AlarmHandler()
current_settings   = Settings()
current_data       = GatheredOverallData()

ina219_i2c = machine.I2C(0, scl=machine.Pin(1), sda=machine.Pin(0))
camera_i2c = machine.I2C(0, scl=machine.Pin(1), sda=machine.Pin(0), freq=400_000)
bme688_i2c = PimoroniI2C(0, 1)
my_bme688  = BreakoutBME68X(bme688_i2c,0x76)
my_display = Display(DISPLAY_INKY_PACK, current_data, alarm_handler_inst, current_settings)
my_camera  = MLX90641(camera_i2c)

my_ina219 = INA219(ina219_i2c, addr=0x45) 

my_ina219.set_calibration_16V_400mA()


display_button_a = Pin(12, Pin.IN, Pin.PULL_UP)
display_button_b = Pin(13, Pin.IN, Pin.PULL_UP)
display_button_c = Pin(14, Pin.IN, Pin.PULL_UP)

sensors_timer = machine.Timer(-1)

camera_trigger_pin.value(0)  # Turn off camera power after init
print("hi")
def update_timers():

    sensors_timer.init(
        period=SENSOR_READ_PERIOD,
        mode=machine.Timer.PERIODIC,
        callback=lambda t: sensors_read_data()
    )

update_timers()

def sensors_read_data():
    machine.freq(CPU_FREQ_NORMAL)
    
    global ina219_counter
    global bme_start_counter
    camera_trigger_pin.value(1)

    time.sleep_ms(CAMERA_DELAY_AFTER_POWER_ON)

    my_camera.get_frame(current_data.camera_data_array)

    if ina219_counter > INA219_COUNTER_VAL:
        current_data.ina219_current = my_ina219.current *10
        current_data.ina219_voltage = my_ina219.bus_voltage
        current_data.battery_percent = convert_voltage_to_percent(current_data.ina219_voltage)
        ina219_counter = 0

    ina219_counter += 1

    if (bme_start_counter < BME688_FIRST_WAKEUP):
        bme_start_counter =+1
        alarm_handler_inst.bme_gas_detected = 0 
        alarm_handler_inst.bme_temp_detected  = 0
    else:
        current_data.bme680_temperature, current_data.bme680_pressure, current_data.bme680_humidity, current_data.bme680_gas_resistance, current_data.bme680_status, _, _ = my_bme688.read()

    camera_trigger_pin.value(0)
    if current_data.bme680_gas_resistance < current_settings.gas_threshold:
        current_data.bme680_gas_calculated = "ALARM"
    elif current_data.bme680_gas_resistance < BME688_MID_QUALITY_LEVEL:
        current_data.bme680_gas_calculated = "WARNING"
    else:
        current_data.bme680_gas_calculated = "SAFE"

    alarm_handler_inst.ir_camera_fire_detected = catch_fire(
        current_data.camera_data_array,
        current_settings.catch_fire_min_temperature,
        current_settings.catch_fire_min_detection_size,
        current_settings.catch_fire_alarm_limit
    )
    current_data.temperature = current_data.bme680_temperature

    if current_data.ina219_current < 0: 
        current_data.is_charging = True
    else:
        current_data.is_charging = False
    
    print(f'Camera_fire:{alarm_handler_inst.ir_camera_fire_detected}')
    print_thermal_frame(current_data.camera_data_array)
    print_frame_grid(current_data.camera_data_array)
    if current_data.bme680_gas_resistance < current_settings.gas_threshold:
        alarm_handler_inst.bme_gas_detected = 1
    if current_data.bme680_temperature > current_settings.bme_temp_threshold:
        alarm_handler_inst.bme_temp_detected = 1

    
    alarm_handler_inst.process_data(current_settings.detection_mode)

    print("Voltage: {}".format(current_data.ina219_voltage))
    print("Current: {} mA".format(current_data.ina219_current))
    print("BME gas resistance: {}".format(current_data.bme680_gas_resistance))
    print("BME temperature: {}".format(current_data.bme680_temperature))

    if alarm_handler_inst.is_alarm:
        my_display.display_alarm_screen()
        buzzer.value(0)
    else:
        buzzer.value(1)
        my_display.current_screen = "main"
    
    print(f'fire detected {alarm_handler_inst.is_alarm}')
    my_display.refresh_screen()
    machine.freq(CPU_FREQ_SLEEP)
   


def button_b_handler(pin):

    machine.freq(CPU_FREQ_NORMAL)
    if my_display.current_screen == "alarm":
        alarm_handler_inst.alarm_callback_off()
        buzzer.value(1)
        my_display.current_screen = "main"
        my_display.refresh_screen()
    elif my_display.current_screen == "main":
        my_display.current_screen = "settings"
        my_display.setting_index = 0
        my_display.refresh_screen()
    elif my_display.current_screen == "settings":
        if my_display.setting_index == 2:
            my_display.just_finished_settings = True
            my_display.refresh_screen()
        else:
            my_display.setting_index += 1
            my_display.refresh_screen()
    else:
        my_display.current_screen = "main"
        my_display.refresh_screen()
    machine.freq(CPU_FREQ_SLEEP)

def button_a_handler(pin):
    machine.freq(CPU_FREQ_NORMAL)
    if my_display.current_screen == "alarm":
        alarm_handler_inst.alarm_callback_off()
        buzzer.value(1)
        my_display.current_screen = "main"
        my_display.refresh_screen()
    elif my_display.current_screen == "settings":
        if my_display.setting_index == 0:
            current_settings.gas_threshold = max(1, current_settings.gas_threshold - 1000)
        elif my_display.setting_index == 1:
            current_settings.fire_threshold = max(1, current_settings.fire_threshold - 5)
        elif my_display.setting_index == 2:
            current_settings.room_volume = max(1, current_settings.room_volume - 1)
        my_display.refresh_screen()
    machine.freq(CPU_FREQ_SLEEP)

def button_c_handler(pin):
    machine.freq(CPU_FREQ_NORMAL)
    if my_display.current_screen == "alarm":
        alarm_handler_inst.alarm_callback_off()
        buzzer.value(1)
        my_display.current_screen = "main"
        my_display.refresh_screen()
    elif my_display.current_screen == "settings":
        if my_display.setting_index == 0:
            current_settings.gas_threshold = current_settings.gas_threshold + 1000
            my_display.refresh_screen()
        elif my_display.setting_index == 1:
            current_settings.fire_threshold = current_settings.fire_threshold + 5
            my_display.refresh_screen()
        elif my_display.setting_index == 2:
            current_settings.room_volume =  current_settings.room_volume + 1
        my_display.refresh_screen()
    machine.freq(CPU_FREQ_SLEEP)
      

display_button_b.irq(trigger=Pin.IRQ_FALLING, handler=button_b_handler)
display_button_a.irq(trigger=Pin.IRQ_FALLING, handler=button_a_handler)
display_button_c.irq(trigger=Pin.IRQ_FALLING, handler=button_c_handler)

  # Save power instead of sleep (sleep not working properly)

machine.freq(CPU_FREQ_SLEEP)

while True:

    time.sleep(1)  # Better than idle
    
