from picographics import PicoGraphics, DISPLAY_INKY_PACK, PEN_1BIT

class Display:
    def __init__(self, display_type, current_data, alarm_handler, settings):

        self.current_data           = current_data
        self.display_type           = display_type
        self.settings               = settings
        self.pen_type               = PEN_1BIT
        self.update_speed           = 2
        self.alarm_handler          = alarm_handler
        self.display                = self.create_display()
        self.current_screen         = "main"
        self.setting_index          = 0
        self.just_finished_settings = False

    def create_display(self):
        try:
            display = PicoGraphics(display=self.display_type, pen_type=self.pen_type)
        except:
            print("error during initialization of display")

        display.set_font("bitmap8")
        display.set_update_speed(self.update_speed)
        return display

    def display_alarm_screen(self):
        self.current_screen = "alarm"
        self.display.set_pen(15)  # Clear screen to white (1-bit inverted logic)
        self.display.clear()

        self.display.set_pen(0)  # Set pen to black
        self.display.text("!!! ALARM !!!", 30, 20, scale=4)
        self.display.text("DANGER DETECTED", 10, 60, scale=2)
        self.display.update()

    def display_thermal_image(self, x_offset=0, y_offset=0, pixel_size=2):
        temp_min = 15.0
        temp_max = 50.0
        temp_range = temp_max - temp_min

        for row in range(12):
            for col in range(16):
                i = row * 16 + col
                temp = self.current_data.camera_data_array[i]

                norm = (temp - temp_min) / temp_range
                norm = max(0, min(norm, 1))

                grayscale = int(norm * 15)

                grayscale = max(0, min(grayscale, 15))

                self.display.set_pen(grayscale)
                self.display.rectangle(
                    x_offset + col * pixel_size,
                    y_offset + row * pixel_size,
                    pixel_size,
                    pixel_size
                )

    def display_main_screen(self):
        charging_status = "Charging: " if self.current_data.is_charging else "          "
        self.current_screen = "main"
        self.display.set_pen(15)
        self.display.clear()
        self.display.set_pen(0)
        self.display.text(f"Temp: {self.current_data.temperature:.2f}C", 160, 100, scale=2)

        self.display.text(f"{charging_status}{self.current_data.battery_percent}%", 160, 40, scale=2)
        self.display.text(f'Smoke: {self.current_data.bme680_gas_calculated}', 160, 70, scale=2)
        self.draw_battery_icon(250, 5, self.current_data.battery_percent)
        self.display_thermal_image(0,0,7)
        self.display.update()

    def draw_battery_icon(self, x, y, percent):
        
        width  = 30
        height = 15
        bar_width = int((width - 4) * (100 - percent) / 100)
        self.display.rectangle(x, y, width, height)
        self.display.rectangle(x + width, y + height // 3, 3, height // 3)
        if (100 - percent) > 0:
            self.display.set_pen(15)
            self.display.rectangle(x + 2, y + 2, bar_width, height - 4)

    def display_transition_alarm_screen(self):
        self.current_screen = "transition_alarm"
        self.display.set_pen(15)
        self.display.clear()
        self.display.set_pen(0)
        self.display.text("Settings saved", 30, 30, scale=3)
        self.display.text("ALARM READY", 40, 70, scale=3)
        self.display.update()

        self.just_finished_settings = False
        self.current_screen = "main"
        self.refresh_screen()

    def display_settings_screen(self):
        self.current_screen = "settings"
        self.display.set_pen(15)
        self.display.clear()
        self.display.set_pen(0)

        y_positions = [10, 40, 70]
        labels = [
            f"Smoke sensitivity: {self.settings.gas_threshold}",
            f"Heat sensitivity: {self.settings.fire_threshold}",
            f"Room size: {self.settings.room_volume}m2"
        ]

        for i, (y, label) in enumerate(zip(y_positions, labels)):
            prefix = "-> " if i == self.setting_index else "   "
            self.display.text(prefix + label, 10, y, scale=2)

        self.display.text("-", 270, 8, scale=4)
        self.display.text("+", 270, 85, scale=4)

        self.display.update()

    def refresh_screen(self):
        if self.alarm_handler.is_alarm:
            self.display_alarm_screen()
        elif self.just_finished_settings:
            self.display_transition_alarm_screen()
        else:
            if self.current_screen == "main":
                self.display_main_screen()
            elif self.current_screen == "settings":
                self.display_settings_screen()
